<?php
/**
 * Plugin Name: Hello 3D Gallery Tabs Fullscreen
 * Description: Display categorized 3D models with tabs, fullscreen, AR, and screenshots — powered by <model-viewer>.
 * Version: 1.0.0
 * Author: Towfique Ar Rahman
 * License: GPLv2 or later
 * Text Domain: hello-3d-gallery-tabs-fullscreen
  * Plugin URI: https://mishudev.com/contact/
  * Author URI: https://mishudev.com/
  * License URI: https://www.gnu.org/licenses/gpl-2.0.html
  * Domain Path: /languages
 */
if (!defined('ABSPATH')) exit;
class H3DG_Plugin {
  const VER='1.1.0'; const H='h3dg';
  function __construct(){
    add_action('init',[$this,'register_types']);
    add_action('add_meta_boxes',[$this,'metabox']);
    add_action('save_post',[$this,'save_meta']);
    add_action('admin_enqueue_scripts',[$this,'admin_assets']);
    add_action('wp_enqueue_scripts',[$this,'front_assets']);
    add_filter('script_loader_tag',[$this,'model_viewer_module'],10,3);
    add_filter('upload_mimes',[$this,'mimes']);
    add_shortcode('three_d_gallery',[$this,'shortcode']);
  }
  function register_types(){
    register_post_type('h3dg_item',[ 'label'=>__('3D Items','h3dg'),'public'=>false,'show_ui'=>true,'menu_icon'=>'dashicons-format-gallery','supports'=>['title','thumbnail','page-attributes'] ]);
    register_taxonomy('h3dg_cat','h3dg_item',[ 'label'=>__('3D Categories','h3dg'),'public'=>false,'show_ui'=>true,'hierarchical'=>false ]);
  }
  function metabox(){ add_meta_box('h3dg_meta',__('3D Model Settings','h3dg'),[$this,'metabox_html'],'h3dg_item','normal','high'); }
  function admin_assets($hook){ if(in_array($hook,['post.php','post-new.php'])){ wp_enqueue_media(); wp_enqueue_style('h3dg-admin',plugin_dir_url(__FILE__).'admin/admin.css',[],self::VER); wp_enqueue_script('h3dg-admin',plugin_dir_url(__FILE__).'admin/admin.js',['jquery'],self::VER,true);} }
  function metabox_html($post){
    $m=get_post_meta($post->ID,'_h3dg_model',true); $p=get_post_meta($post->ID,'_h3dg_poster',true);
    $e=get_post_meta($post->ID,'_h3dg_env',true); $c=get_post_meta($post->ID,'_h3dg_camera',true);
    wp_nonce_field('h3dg_save','h3dg_nonce'); ?>
    <div class="h3dg-fields">
      <p><label><?php _e('Model URL (.glb/.gltf/.usdz)','h3dg'); ?></label>
      <input type="url" name="h3dg_model" value="<?php echo esc_attr($m); ?>" placeholder="https://.../model.glb" />
      <button class="button h3dg-pick" data-target="h3dg_model"><?php _e('Select from Media','h3dg'); ?></button></p>
      <p><label><?php _e('Poster image URL (optional)','h3dg'); ?></label>
      <input type="url" name="h3dg_poster" value="<?php echo esc_attr($p); ?>" placeholder="https://.../poster.jpg" />
      <button class="button h3dg-pick" data-target="h3dg_poster"><?php _e('Select from Media','h3dg'); ?></button></p>
      <div class="h3dg-grid">
        <p><label><?php _e('Environment','h3dg'); ?></label>
        <input type="text" name="h3dg_env" value="<?php echo esc_attr($e?:'neutral'); ?>" placeholder="neutral or HDR URL" /></p>
        <p><label><?php _e('Camera Orbit','h3dg'); ?></label>
        <input type="text" name="h3dg_camera" value="<?php echo esc_attr($c?:'0deg 70deg 1.5m'); ?>" placeholder="0deg 70deg 1.5m" /></p>
      </div>
      <p class="desc"><?php _e('Use Featured Image as the left preview image. Assign "3D Categories" for top tabs.','h3dg'); ?></p>
    </div><?php
  }
  function save_meta($id){
    if(!isset($_POST['h3dg_nonce'])||!wp_verify_nonce($_POST['h3dg_nonce'],'h3dg_save'))return;
    if(defined('DOING_AUTOSAVE')&&DOING_AUTOSAVE)return;
    if(!current_user_can('edit_post',$id))return;
    update_post_meta($id,'_h3dg_model',esc_url_raw($_POST['h3dg_model']??''));
    update_post_meta($id,'_h3dg_poster',esc_url_raw($_POST['h3dg_poster']??''));
    update_post_meta($id,'_h3dg_env',sanitize_text_field($_POST['h3dg_env']??''));
    update_post_meta($id,'_h3dg_camera',sanitize_text_field($_POST['h3dg_camera']??''));
  }
  function front_assets(){
    $u=plugin_dir_url(__FILE__);
    wp_register_style(self::H,$u.'assets/style.css',[],self::VER);
    wp_register_script(self::H,$u.'assets/front.js',['jquery'],self::VER,true);
    wp_register_script('model-viewer','https://unpkg.com/@google/model-viewer/dist/model-viewer.min.js',[],null,true);
  }
  function model_viewer_module($tag,$handle,$src){ if($handle==='model-viewer') $tag='<script type="module" src="'.esc_url($src).'"></script>'; return $tag; }
  function mimes($m){ $m['glb']='model/gltf-binary'; $m['gltf']='model/gltf+json'; $m['bin']='application/octet-stream'; $m['usdz']='model/vnd.usdz+zip'; $m['hdr']='image/vnd.radiance'; return $m; }
  function shortcode($atts){
    $a=shortcode_atts(['cats'=>'','height'=>'520','env'=>'neutral','ar'=>'true','auto_rotate'=>'true','exposure'=>'1.0','shadow'=>'0.2','fov'=>'30deg','orbit'=>'0deg 70deg 1.5m'],$atts,'three_d_gallery');
    $targs=['taxonomy'=>'h3dg_cat','hide_empty'=>true]; if(!empty($a['cats'])) $targs['slug']=array_map('sanitize_title', array_map('trim', explode(',',$a['cats'])));
    $terms=get_terms($targs); if(is_wp_error($terms)||empty($terms)) return '<div class="h3dg-notice">No 3D categories/items found.</div>';
    $cats=[]; $first=null;
    foreach($terms as $t){
      $q=new WP_Query(['post_type'=>'h3dg_item','posts_per_page'=>-1,'orderby'=>'menu_order title','order'=>'ASC','tax_query'=>[[ 'taxonomy'=>'h3dg_cat','field'=>'term_id','terms'=>$t->term_id ]]]);
      $items=[]; while($q->have_posts()){ $q->the_post(); $id=get_the_ID();
        $it=['label'=>get_the_title(),'thumb'=>get_the_post_thumbnail_url($id,'medium')?:'','model'=>get_post_meta($id,'_h3dg_model',true),'poster'=>get_post_meta($id,'_h3dg_poster',true),'env'=>get_post_meta($id,'_h3dg_env',true)?:$a['env'],'camera'=>get_post_meta($id,'_h3dg_camera',true)?:$a['orbit']];
        if(!$first) $first=$it; $items[]=$it;
      } wp_reset_postdata();
      if($items) $cats[]=['slug'=>$t->slug,'name'=>$t->name,'items'=>$items];
    }
    if(!$cats) return '<div class="h3dg-notice">No 3D items in selected categories.</div>';
    wp_enqueue_style(self::H); wp_enqueue_script('model-viewer'); wp_enqueue_script(self::H);
    $h=preg_replace('/[^0-9.]/','',$a['height']); $env=esc_attr($a['env']); $ar=filter_var($a['ar'],FILTER_VALIDATE_BOOLEAN)?'ar':''; $auto=filter_var($a['auto_rotate'],FILTER_VALIDATE_BOOLEAN)?'auto-rotate':''; $exp=esc_attr($a['exposure']); $sh=esc_attr($a['shadow']); $fov=esc_attr($a['fov']);
    ob_start(); ?>
<div class="h3dg" data-h3dg data-height="<?php echo esc_attr($h); ?>">
  <div class="h3dg-tabs" role="tablist" aria-label="3D categories">
    <?php foreach($cats as $i=>$c): ?><button class="h3dg-tab<?php echo $i===0?' active':''; ?>" data-cat="<?php echo esc_attr($c['slug']); ?>" role="tab" aria-selected="<?php echo $i===0?'true':'false'; ?>"><?php echo esc_html($c['name']); ?></button><?php endforeach; ?>
  </div>
  <div class="h3dg-body">
    <div class="h3dg-left">
      <?php foreach($cats as $i=>$c): ?><div class="h3dg-list" data-cat="<?php echo esc_attr($c['slug']); ?>" style="display:<?php echo $i===0?'block':'none'; ?>">
        <?php foreach($c['items'] as $j=>$it): ?><div class="h3dg-card<?php echo ($i===0&&$j===0)?' active':''; ?>" role="button" tabindex="0"
          data-model="<?php echo esc_url($it['model']); ?>" data-poster="<?php echo esc_url($it['poster']); ?>" data-env="<?php echo esc_attr($it['env']); ?>" data-camera="<?php echo esc_attr($it['camera']); ?>">
          <?php if($it['thumb']): ?><img src="<?php echo esc_url($it['thumb']); ?>" alt="" /><?php endif; ?>
        </div><?php endforeach; ?>
      </div><?php endforeach; ?>
    </div>
    <div class="h3dg-right">
      <model-viewer class="h3dg-mv"
        src="<?php echo esc_url($first['model']); ?>" <?php if(!empty($first['poster'])): ?>poster="<?php echo esc_url($first['poster']); ?>"<?php endif; ?>
        <?php echo $ar; ?> ar-modes="webxr scene-viewer quick-look" loading="lazy" reveal="auto"
        camera-controls touch-action="pan-y" <?php echo $auto; ?>
        environment-image="<?php echo $env; ?>" exposure="<?php echo $exp; ?>" shadow-intensity="<?php echo $sh; ?>"
        camera-orbit="<?php echo esc_attr($first['camera']); ?>" field-of-view="<?php echo $fov; ?>" interaction-prompt="none">
        <div slot="progress-bar" class="h3dg-progress"><div class="h3dg-progress-inner"><span class="h3dg-progress-text">Loading…</span></div></div>
      </model-viewer>
      <div class="h3dg-toolbar">
        <button type="button" class="btn-reset">Reset</button>
        <button type="button" class="btn-screenshot">Screenshot</button>
        <button type="button" class="btn-ar">AR</button>
        <button type="button" class="btn-fullscreen">Fullscreen</button>
      </div>
    </div>
  </div>
</div>
<script type="application/json" class="h3dg-config"><?php echo wp_json_encode(['env'=>$env,'exposure'=>$exp,'shadow'=>$sh,'fov'=>$fov,'orbit'=>$a['orbit']]); ?></script>
<?php return ob_get_clean();
  }
}
new H3DG_Plugin();

